"""pyLCD.py, Python module to control a HD44780 based (or compatible) character type LCD.
"""
# Version 1.0, 04.11.2001
# Requires:
#     winioport
#     Windows
# Usage:
# >>> import pyLCD
# >>> pyLCD.fs1line()     # 1-line, 8-bit interface
# >>> pyLCD.ems()         # entry mode set (default: increase cursor position, no shift )
# >>> pyLCD.dispOn()      # display on with cusrsor invisible
# >>> pyLCD.cursHome()    # cursor home 
# >>> pyLCD.putstr('Hello World') # print the string
# >>>

# LCD is supposed to be connected to the printer port as shown below.
# Default port address is 0x378. It can be changed in the winioport.py file

# LCD connections                                                                                                                                         
  
# D0           D0                                                                                                                                          
# D1           D1                                                                                                                                          
# D2           D2                                                                                                                                          
# D3           D3                                                                                                                                          
# D4           D4                                                                                                                                          
# D5           D5                                                                                                                                          
# D6           D6                                                                                                                                          
# D7           D7                                                                                                                                          
  
# E            Initialize ... (pin 16)   ctrlreg_Bit2                                                                                                       
# R/W          Tied to ground >> No busy flag check or any other read from display will be done.                                                           
# RS           Strobe ....... (pin 1)   ctrlreg_Bit0    

# Copyright (c) 2001, Dincer Aydin
# dinceraydin@yahoo.com
# www.geocities.com/dinceraydin/python/

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY 
# KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
# THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
# PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
# DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
# CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN 
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
# IN THE SOFTWARE.
#                                                                                                 


from  winioport import *
import time

def delay(): time.sleep(0.1)    # 0.1 second delay
def enable(): pportInitOut(1)   # E goes high.
def disable(): pportInitOut(0)  # E goes low.
#def read(): pportAutoFeed(1)    # RW goes high -> read from display
def write(): pportAutoFeed(0)   # RW goes low -> write to display
def instrreg(): pportDataStrobe(0)  # RS goes low -> DATA register selected
def datareg(): pportDataStrobe(1)   # RS goes high -> INSTRUCTION register selected


def instr(commnd):
    "send instruction to LCD"
    disable()
    write()
    instrreg()
    pportOut(commnd)
    enable()
    disable()

def data(char):
    "send data to LCD"
    disable()
    write()
    datareg()
    pportOut(char)
    enable()
    disable()
    
def fs1line():
    "8-bit interface 1-line display"
    instr(48)
    
def fs2line():
    "8-bit interface 2-line display"
    instr(56)

def sdl(times = 1):
    "Shift display left."
    for i in range(times):
        instr(24)
        
def sdr(times = 1):
    "Shift display right."
    for i in range(times):
        instr(28)

def mcl(times = 1):
    "Move cursor left."
    for i in range(times):
        instr(16)

def mcr(times = 1):
    "Move cursor right."
    for i in range(times):
        instr(20)
        
def blockCurs():
    "Display ON with block cursor."
    instr(15)

def undCurs():
    "Display ON with underline cursor."
    instr(15)
    
def cursOff():
    "Display ON without cursor."
    instr(15)

def dispOn():
    "Display ON without cursor."
    instr(12)

def blankDisp():
    "Display OFF."
    instr(8)

def cursHome():
    "Cursor home."
    delay()
    instr(2)
    delay()
    delay()
    
def clear():
    "Clear display."
    delay()
    instr(1)
    delay()
    delay()

def ddradrs(adrs):
    "Set DDRAM address."
    instr(128 + adrs)

def cgradrs(adrs):
    "Set CGRAM address."
    instr(64 + adrs)

def ems(incr = 1, shift = 0): # default: increase cursor position, no shift 
    "Entry mode set."
    cmmnd = 0x4
    if incr: cmmnd |= 0x2
    if shift: cmmnd |= 0x1
    instr(cmmnd)

def putstr(strng):
    for i in strng:
        data(ord(i))


